<?php

namespace App\Http\Controllers\API;

use App\Events\EmailChanged;
use App\Http\Requests\AliasRequest;
use App\Http\Requests\BankingDataRequest;
use App\Http\Requests\ChangePasswordRequest;
use App\Http\Requests\CompanyAddressRequest;
use App\Http\Requests\CompanyRequest;
use App\Services\UserService;
use App\Services\VerificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Validator;
use Illuminate\Support\Facades\Auth;
use App\Services\AgentRegistrationService;
use App\User;

class UserController extends Controller
{
    /**
     * Method to login user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login()
    {
        if (Auth::attempt(['email' => request('email'), 'password' => request('password')])) {
            Auth::user()->generateToken();
            return $this->_s(Auth::user());
        } else {
            return $this->_e('Der Benutzername oder das Passwort ist falsch', 401);
        }
    }

    /**
     * Method to log out current user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        $user = Auth::user();
        if ($user) {
            $user->api_token = null;
            $user->save();

            Auth::logout();
        }

        return $this->_s();
    }

    /**
     * Method to register new user.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'bail', 'min:10', 'password_strength:2'],
        ], [
            'password.password_strength' => 'Das Passwort ist zu schwach.',
        ]);

        if ($validator->fails()) {
            return $this->_e($validator->errors());
        }

        $seller = User::where('id', $request->seller)->first();

        $data = $request->all();
        $registrationService = new AgentRegistrationService($data);
        $user = $registrationService->register();
        $registrationService->sendDoubleOptInMail($user);

        if ($seller) {
            $registrationService->connectAgentWithSeller($seller);
        }

        if ('widget' === $request->get('source', 'widget')) {
            Auth::login($user);
            $user->generateToken();
            return $this->_s(Auth::user());
        }

        return $this->_s();
    }

    /**
     * Method to check user is logged in.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function isLogged()
    {
        if (Auth::user()) {
            return $this->_s(Auth::user());
        } else {
            return $this->_e('Unauthorised', 401);
        }
    }


    /**
     * Method to change user password.
     *
     * @param ChangePasswordRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changePassword(ChangePasswordRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $user = Auth::user();
        $currentPassword = $user->password;
        if ( Hash::check($requestData['old'], $currentPassword) ) {
            $user->password = Hash::make($requestData['new']);
            $user->save();

            activity($user)->register('password changed');

            return $this->_s();
        } else {
            return $this->_e(['old' => ['Das Altes Passwort ist nicht richtig']]);
        }
    }

    /**
     * Method to change user address.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeAddress(Request $request)
    {
        $requestData = $this->getRequestData($request);

        $validator = Validator::make($requestData, [
            'street' => ['required', 'max:100'],
            'housenumber' => ['required', 'max:20'],
            'zip' => ['required', 'regex:/^([0]{1}[1-9]{1}|[1-9]{1}[0-9]{1})[0-9]{3}$/'],
            'city' => ['required', 'max:100'],
        ], [], [
            'street' => 'Straße',
            'housenumber' => 'Hausnummer',
            'zip' => 'PLZ',
            'city' => 'Ort',
        ]);

        if ($validator->fails()) {
            return $this->_e($validator->errors());
        }

        $user = Auth::user();
        $user->account->street = $requestData['street'];
        $user->account->housenumber = $requestData['housenumber'];
        $user->account->zip = $requestData['zip'];
        $user->account->city = $requestData['city'];
        $user->account->save();

        return $this->_s();
    }

    /**
     * Method to change company address.
     *
     * @param CompanyAddressRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeCompanyAddress(CompanyAddressRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $user   = Auth::user();
        $seller = Auth::user();

        if ( $user->hasRole('employee') ) {
            $seller = $user->boss;
        }

        $oldData = [
            'street'      => $seller->account->street,
            'housenumber' => $seller->account->housenumber,
            'zip'         => $seller->account->zip,
            'city'        => $seller->account->city,
            'email'       => $seller->account->email,
        ];

        $seller->account->street      = $requestData['street'];
        $seller->account->housenumber = $requestData['housenumber'];
        $seller->account->zip         = $requestData['zip'];
        $seller->account->city        = $requestData['city'];
        $seller->account->email       = $requestData['email'];
        $seller->account->save();

        if (isset($requestData['phone'])) {
            $oldData['phone'] = $seller->phone;
            $seller->phone = $requestData['phone'];
            $seller->save();
        }

        activity($user)->register('company address edited', $oldData, $requestData);

        return $this->_s();
    }

    /**
     * Method to change user personal data.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changePersonal(Request $request)
    {
        $requestData = $this->getRequestData($request);

        $validator = Validator::make($requestData, [
            'salutation' => ['required', 'in:Herr,Frau'],
            'firstname' => ['required', 'max:50'],
            'surname' => ['required', 'max:50'],
        ], [], [
            'salutation' => 'Anrede',
            'firstname' => 'Vorname',
            'surname' => 'Nachname',
        ]);

        if ($validator->fails()) {
            return $this->_e($validator->errors());
        }

        $user = Auth::user();

        $oldData = [
            'salutation' => $user->salutation,
            'firstname'  => $user->firstname,
            'surname'    => $user->surname,
        ];

        $user->salutation = $requestData['salutation'];
        $user->firstname  = $requestData['firstname'];
        $user->surname    = $requestData['surname'];
        $user->save();

        activity($user)->register('private data edited', $oldData, $requestData);

        return $this->_s();
    }

    /**
     * Method to process user email.
     *
     * @param $user
     * @param $requestData
     */
    private function _processEmail($user, $requestData)
    {
        if ($user->email_temp && $user->email_temp === $requestData['email']) {
            // do nothing (token already sent)
        } else {
            if ($user->email_valid === $requestData['email'] && $user->email_temp) { // email is equals to old approved
                $user->email = $requestData['email'];
                $user->email_temp = '';
                $user->email_verified = true;
                $user->email_verification_token = null;
            } else { // email changed
                $user->email_temp = $requestData['email'];
            }
            $user->save();
            event(new EmailChanged($user));
        }
    }

    /**
     * Method to change user contacts data.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeContacts(Request $request)
    {
        $requestData = $this->getRequestData($request);

        $user = Auth::user();

        $validator = Validator::make($requestData, [
            'email' => ['required', 'email', 'unique:users,email,' . $user->id],
        ], [], [
            'email' => 'E-Mail',
        ]);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $user->phone = $requestData['phone'];
        $user->save();

        $this->_processEmail($user, $requestData);

        return $this->_s();
    }

    /**
     * Method to change users email.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeEmail(Request $request)
    {
        $requestData = $this->getRequestData($request);

        $user = Auth::user();
        $oldEmail = $user->email;

        $validator = Validator::make($requestData, [
            'email' => ['required', 'email', 'unique:users,email,' . $user->id],
        ], [], [
            'email' => 'E-Mail',
        ]);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $this->_processEmail($user, $requestData);

        activity($user)->register('email edited', ['email' => $oldEmail], $requestData);

        return $this->_s();
    }

    /**
     * Method to change user banking data.
     *
     * @param BankingDataRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeBanking(BankingDataRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $user   = Auth::user();
        $seller = Auth::user();

        if ( $user->hasRole('employee') ) {
            $seller = $user->boss;
        }

        $oldData = [
            'iban'       => $seller->account->iban,
            'iban_owner' => $seller->account->iban_owner,
        ];

        $seller->account->iban       = $requestData['iban'];
        $seller->account->iban_owner = $requestData['iban_owner'];
        $seller->account->save();

        activity($user)->register('banking edited', $oldData, $requestData);

        return $this->_s();
    }

    /**
     * Method to change user alias.
     *
     * @param AliasRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeAlias(AliasRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $user   = Auth::user();
        $seller = Auth::user();

        if ( $user->hasRole('employee') ) {
            $seller = $user->boss;
        }

        $oldAlias = $seller->alias;

        $seller->alias = $requestData['alias'];
        $seller->save();

        activity($user)->register('alias edited', ['alias' => $oldAlias], $requestData);

        return $this->_s();
    }

    /**
     * Method to change company data.
     *
     * @param CompanyRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeCompany(CompanyRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $user   = Auth::user();
        $seller = Auth::user();

        if ( $user->hasRole('employee') ) {
            $seller = $user->boss;
        }

        $oldData = [
            'company'     => $seller->account->company,
            'description' => $seller->account->description,
        ];

        $seller->account->company     = $requestData['company'];
        $seller->account->description = $requestData['description'];
        $seller->account->save();

        activity($user)->register('company edited', $oldData, $requestData);

        return $this->_s();
    }

    /**
     * Method to change company worktimes.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeWorktimes(Request $request)
    {
        $requestData = $this->getRequestData($request);

        $user = Auth::user();
        $user->account->worktimes = json_encode([
            $requestData['wt0'],
            $requestData['wt1'],
            $requestData['wt2'],
            $requestData['wt3'],
            $requestData['wt4'],
            $requestData['wt5'],
            $requestData['wt6'],
        ]);
        $user->account->save();

        return $this->_s();
    }

    /**
     * Method to remove current user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function remove()
    {
        $user = Auth::user();

        (new UserService($user))->delete();

        Auth::logout();

        return $this->_s();
    }

    /**
     * Method to send email verification token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function sendEmailVerificationToken()
    {
        $user = Auth::user();

        (new VerificationService($user))->sendEmailVerification();

        return $this->_s();
    }
}