<?php

namespace App\Http\Controllers\API;

use App\CouponTemplate;
use App\Http\Requests\RegisterShareRequest;
use App\Http\Requests\ShareLinkRequest;
use App\Notifications\Mail\ShareInviteLinkNotification;
use App\Notifications\Mail\ShareReferralLinkNotification;
use App\User;

class ShareController extends Controller
{
    /**
     * Method to get seller by deal, seller id or user.
     *
     * @param $requestData
     * @return null
     */
    private function getSeller($requestData)
    {
        if ( isset($requestData['entity']) && 'deal' === $requestData['entity'] ) {
            $deal = CouponTemplate::find($requestData['entity_id']);
            if ( null !== $deal ) {
                return $deal->seller;
            }
        } else if ( isset($requestData['entity']) && 'seller' === $requestData['entity'] ) {
            return User::find($requestData['entity_id']);
        } else if ( $this->user->hasRole('employee') ) {
            return $this->user->boss;
        } else if ( $this->user->hasRole('seller') ) {
            return $this->user;
        } else {
            return $this->user->getAttachedSeller();
        }

        return null;
    }

    /**
     * Method to send referral link via email.
     *
     * @param ShareLinkRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function shareReferralLinkViaEmail(ShareLinkRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $seller = $this->getSeller($requestData);

        $templateData = array_merge(
            $requestData,
            [
                'user'   => $this->user,
                'seller' => $seller,
            ]
        );

        $subject = (null !== $seller)
            ? $this->user->fullname . ' empfiehlt Dir ' . $seller->account->company . ', hol Dir jetzt Dein Cashback'
            : $this->user->fullname . ' empfiehlt Dir RECO.MA, hol Dir jetzt Dein Cashback';

        (new ShareReferralLinkNotification())
            ->setSubject($subject)
            ->setTemplateData($templateData)
            ->setRecipient($requestData['email'])
            ->setReply($this->user->email, $this->user->fullname)
            ->send();

        return $this->_s();
    }

    /**
     * Method to send seller invite link via email.
     *
     * @param ShareLinkRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function shareInviteLinkViaEmail(ShareLinkRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $seller = $this->getSeller($requestData);

        $inviter = ( $this->user->hasRole('seller') || $this->user->hasRole('employee') )
            ? $seller->account->company
            : $this->user->fullname;

        $templateData = array_merge(
            $requestData,
            [
                'inviter' => $inviter,
            ]
        );

        $subject = $inviter . ' lädt Dich zu RECO.MA ein, Deinem persönlichen Empfehlungsportal';

        (new ShareInviteLinkNotification())
            ->setSubject($subject)
            ->setTemplateData($templateData)
            ->setRecipient($requestData['email'])
            ->setReply($this->user->email, $this->user->fullname)
            ->send();

        return $this->_s();
    }

    /**
     * Method to register sharing of the entity.
     *
     * @param RegisterShareRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registerShare(RegisterShareRequest $request)
    {
        $requestData = $this->getRequestData($request);

        $entity = ('deal' === $requestData['entity'])
            ? CouponTemplate::find($requestData['id'])
            : User::find($requestData['id']);

        shares($entity)->register($requestData['provider']);

        return $this->_s();
    }
}