<?php

namespace App\Http\Controllers\API;

use App\Shortlink;
use Illuminate\Http\Request;
use Validator;
use App\Services\LinkInvitationService;
use App\User;
use App\CouponTemplate;

class InvitationController extends Controller
{
    /**
     * Method to share seller via email.
     *
     * @param User $seller
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function shareSeller(User $seller, Request $request)
    {
        $requestData = $this->getRequestData($request);

        if ( !$this->user->hasRole('agent') ) {
            return $this->_e('Nur der Benutzer kann den Verkäufer teilen', 401);
        }

        $validator = Validator::make($requestData, [
            'email' => 'required|email|unique:users,email',
        ]);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $linkInvitationService = new LinkInvitationService($this->user);
        $linkInvitation = $linkInvitationService->create('share-seller', null, $seller->account->id);
        $url = url('/registration/' . $linkInvitation->token);

        $data = $requestData;
        $data['sellerAccount'] = $seller->account;
        $data['agent'] = $this->user;
        $data['url'] = $url;
        $data['emailText'] = '';
        $this->userData = [
            'firstname' => '',
            'surname'   => '',
            'email'     => $requestData['email'],
        ];
        $toUser = new User();
        $toUser->fill($this->userData);
        $toUser->sendMail('agent-sends-coupon-to-new-consumer', $data, $this->user);

        return $this->_s();
    }

    /**
     * Method to generate share seller link.
     *
     * @param User $seller
     * @return \Illuminate\Http\JsonResponse
     */
    public function getShareSellerLink(User $seller)
    {
        if ( !$this->user->hasRole('agent') ) {
            return $this->_e('Nur der Benutzer kann den Verkäufer teilen', 401);
        }

        $linkInvitationService = new LinkInvitationService($this->user);
        $linkInvitation = $linkInvitationService->create('share-seller', null, $seller->account->id);
        $url = url('/registration/' . $linkInvitation->token);

        $link = Shortlink::shorten($url);

        return $this->_s($link);
    }

    /**
     * Method to share coupon via email.
     *
     * @param CouponTemplate $couponTemplate
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function shareCoupon(CouponTemplate $couponTemplate, Request $request)
    {
        $requestData = $this->getRequestData($request);

        if ( !$this->user->hasRole('agent') ) {
            return $this->_e('Nur der Benutzer kann den Deal teilen', 401);
        }

        $validator = Validator::make($requestData, [
            'email' => 'required|email|unique:users,email',
        ]);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $linkInvitationService = new LinkInvitationService($this->user);
        $linkInvitation = $linkInvitationService->create('share-coupon', $couponTemplate->id);
        $url = url('/registration/' . $linkInvitation->token);

        $data = $requestData;
        $data['sellerAccount'] = $couponTemplate->seller->account;
        $data['agent'] = $this->user;
        $data['url'] = $url;
        $data['emailText'] = '';
        $this->userData = [
            'firstname' => '',
            'surname'   => '',
            'email'     => $requestData['email'],
        ];
        $toUser = new User();
        $toUser->fill($this->userData);
        $toUser->sendMail('agent-sends-coupon-to-new-consumer', $data, $this->user);

        return $this->_s();
    }

    /**
     * Method to generate share coupon link.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\JsonResponse
     */
    public function getShareCouponLink(CouponTemplate $couponTemplate)
    {
        if ( !$this->user->hasRole('agent') ) {
            return $this->_e('Nur der Benutzer kann den Deal teilen', 401);
        }

        $linkInvitationService = new LinkInvitationService($this->user);
        $linkInvitation = $linkInvitationService->create('share-coupon', $couponTemplate->id);
        $url = url('/registration/' . $linkInvitation->token);

        $link = Shortlink::shorten($url);

        return $this->_s($link);
    }
}