<?php

namespace App\Http\Controllers\API;

use App\CouponTemplate;
use App\Http\Requests\CouponTemplateRequest;
use App\Rating;
use App\Services\BarCodeService;
use App\Services\DealService;
use Illuminate\Http\Request;
use Validator;

class DealController extends Controller
{
    /**
     * Method to attach new coupon to current user.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCashback(CouponTemplate $couponTemplate)
    {
        if ( !$couponTemplate->isActive() ) {
            return $this->_e('Kampagne ist nicht aktiv', 401);
        }

        if ( !$couponTemplate->isShareable() ) {
            return $this->_e('Kampagne ist nicht teilbar', 401);
        }

        if ( !$this->user->hasRole('agent') ) {
            return $this->_e('Nur der Benutzer kann das machen', 401);
        }

        // TODO refactor it, it is very bad code (coupon storing should be in service)
        $coupon = app('App\Http\Controllers\CouponTemplateController')
            ->storeCoupon($couponTemplate->id, $this->user->id, $this->user);

        $coupon->qrcode = (new BarCodeService($coupon->code))->getQRCode();

        return $this->_s($coupon);
    }

    /**
     * Method to update deal.
     *
     * @param CouponTemplateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(CouponTemplateRequest $request)
    {
        $deal = (new DealService(CouponTemplate::find($request->get('deal_id'))))
            ->update($request->all());

        return $this->_s($deal);
    }

    /**
     * Method to rate deal.
     *
     * @param CouponTemplate $couponTemplate
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function rate(CouponTemplate $couponTemplate, Request $request)
    {
        $rules = [
            'rating' => 'required|min:1|max:5',
            'text'   => 'required|min_words:10',
        ];

        $messages = [
            'rating.required'  => 'Wählen Sie die Bewertung.',
            'rating.min'       => 'Wählen Sie die Bewertung.',
            'rating.max'       => 'Wählen Sie die Bewertung.',
            'text.required'    => 'Bewertung muss ausgefüllt sein.',
            'text.min_words'   => 'Bewertung muss mindestens :min_words Wörter haben.',
        ];

        $requestData = $this->getRequestData($request);

        $validator = Validator::make($requestData, $rules, $messages);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $rating = Rating::create([
            'rating'             => $requestData['rating'],
            'user_id'            => $this->user->id,
            'rateable_id'        => $couponTemplate->id,
            'description'        => $requestData['text'],
            'rateable_type'      => CouponTemplate::class,
            'rateable_user_type' => 'consumer',
            'rated_account_id'   => $couponTemplate->seller->account->id,
        ]);

        return $this->_s($rating);
    }

    /**
     * Method to add deal to favorites or remove if it already in.
     *
     * @param CouponTemplate $couponTemplate
     * @return \Illuminate\Http\JsonResponse
     */
    public function favoriteToggle(CouponTemplate $couponTemplate)
    {
        $hasInFav = $this->user->favorites()->where('deal_id', $couponTemplate->id)->exists();

        if ( !$hasInFav ) {
            $this->user->favorites()->attach($couponTemplate);
        } else {
            $this->user->favorites()->detach($couponTemplate);
        }

        return $this->_s();
    }
}