<?php

namespace App\Http\Controllers\API;

use App\Coupon;
use App\CouponTemplate;
use App\Http\Requests\DeleteCoupon;
use App\Http\Requests\MarkCouponAsPaid;
use App\Rating;
use App\Services\CouponService;
use App\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Validator;

class CouponController extends Controller
{
    /**
     * Method to rate coupon.
     *
     * @param Coupon $coupon
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function rate(Coupon $coupon, Request $request)
    {
        if ( $this->user->id !== $coupon->consumer_user_id) {
            return $this->_e('Dieser RECO.BON gehört dir nicht', 401);
        }

        if ( Coupon::STATUS_PAID !== $coupon->status ) {
            return $this->_e('Dieser RECO.BON ist nicht bewertbar', 401);
        }

        $rules = [
            'rating' => 'required|min:1|max:5',
            'text'   => 'required',
        ];

        $messages = [
            'rating.required' => 'Wählen Sie die Bewertung.',
            'rating.min'      => 'Wählen Sie die Bewertung.',
            'rating.max'      => 'Wählen Sie die Bewertung.',
            'text.required'   => 'Bewertung muss ausgefüllt sein.',
        ];

        $requestData = $this->getRequestData($request);

        $validator = Validator::make($requestData, $rules, $messages);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $coupon->status = Coupon::STATUS_DONE;
        $coupon->save();

        Rating::create([
            'rating' => $requestData['rating'],
            'user_id' => $this->user->id,
            'rateable_id' => $coupon->id,
            'description' => $requestData['text'],
            'rateable_type' => Coupon::class,
            'rateable_user_type' => 'consumer',
            'rated_account_id' => $coupon->template->seller->account->id,
        ]);

        $transactions = $coupon->transactions()
            ->where('status', Transaction::STATUS_PENDING)
            ->whereIn('type', Transaction::RELEASEABLE)
            ->get();
        foreach ($transactions as $transaction) {
            $transaction->releaseEligible();
        }

        return $this->_s($coupon);
    }

    /**
     * Method to redeem coupon.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function redeem(Request $request)
    {
        $user = $this->user;

        if ( $user->hasRole('employee') ) {
            if ( !$user->canDo('redeem coupons', true) ) {
                return $this->_e('Forbidden', 403);
            }

            $user = $user->boss;
        }

        if ( !$user->hasRole('seller-active') ) {
            return $this->_e('Nur der Verkäufer kann den RECO.BON einlösen', 401);
        }

        if ( !$user->hasPaymentType() ) {
            return $this->_e('Du hast keine Zahlungsdaten', 401);
        }

        if ( !$user->email_verified ) {
            return $this->_e('Bitte bestätige Deine E-Mail', 401);
        }

        $requestData = $this->getRequestData($request);

        $rules = [
            'code' => 'required|min:9|max:9',
        ];

        $messages = [
            'code.required' => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.min'      => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.max'      => 'RECO.CODE sollte 9 Symbole enthalten',
        ];

        $validator = Validator::make($requestData, $rules, $messages);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $code    = trim(strtoupper($requestData['code']));
        $netto   = (float)$requestData['netto'];
        $payment = $requestData['payment'];

        $coupon = Coupon::where('code', $code)->first();

        if (!$coupon) {
            return $this->_e('Dieser RECO.BON wurde nicht gefunden', 404);
        }

        if ($coupon->template->user_id !== $user->id) {
            return $this->_e('Dieser RECO.BON gehört Dir nicht', 401);
        }

        if (Coupon::STATUS_ACTIVE !== $coupon->status) {
            return $this->_e('Dieser RECO.BON ist schon eingelöst');
        }

        if ( $coupon->validity_date->lt(Carbon::today()) ) {
            return $this->_e('Dieser RECO.BON ist schon abgelaufen');
        }

        if (
            (CouponTemplate::TYPE_PERCENT === $coupon->type_discount || CouponTemplate::TYPE_PERCENT === $coupon->type_provision) &&
            $netto <= 0
        ) {
            return $this->_e('Dieser RECO.BON benötigt eine Warenwert');
        }

        (new CouponService($coupon))->redeem($netto, $payment);

        if ('cash' === $payment) {
            (new CouponService($coupon))->setPaid();
        }

        return $this->_s();
    }

    /**
     * Method to get coupon data by its code.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getByCode(Request $request)
    {
        $user = $this->user;

        if ( $user->hasRole('employee') ) {
            if ( !$user->canDo('redeem coupons', true) ) {
                return $this->_e('Forbidden', 403);
            }

            $user = $user->boss;
        }

        if ( !$user->hasRole('seller-active') ) {
            return $this->_e('Nur der Verkäufer kann den RECO.BON ansehen', 401);
        }

        if ( !$user->hasPaymentType() ) {
            return $this->_e('Du hast keine Zahlungsdaten', 401);
        }

        if ( !$user->email_verified ) {
            return $this->_e('Bitte bestätige Deine E-Mail', 401);
        }

        $requestData = $this->getRequestData($request);

        $rules = [
            'code' => 'required|min:9|max:9',
        ];

        $messages = [
            'code.required' => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.min'      => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.max'      => 'RECO.CODE sollte 9 Symbole enthalten',
        ];

        $validator = Validator::make($requestData, $rules, $messages);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $code = trim(strtoupper($requestData['code']));

        $coupon = Coupon::where('code', $code)
            ->with('agent', 'consumer')
            ->first();

        if (!$coupon) {
            return $this->_e('Dieser RECO.BON wurde nicht gefunden', 404);
        }

        if ($coupon->template->user_id !== $user->id) {
            return $this->_e('Dieser RECO.BON gehört Dir nicht', 401);
        }

        if (Coupon::STATUS_ACTIVE !== $coupon->status) {
            return $this->_e('Dieser RECO.BON ist schon eingelöst');
        }

        if ( $coupon->validity_date->lt(Carbon::today()) ) {
            return $this->_e('Dieser RECO.BON ist schon abgelaufen');
        }

        $result = collect([
            'title'          => $coupon->title,
            'code'           => $coupon->code,
            'agent'          => [
                'id'         => $coupon->agent->id,
                'name'       => $coupon->agent->fullname,
                'userpic'    => $coupon->agent->userpic,
            ],
            'consumer'       => [
                'id'         => $coupon->consumer->id,
                'name'       => $coupon->consumer->fullname,
                'userpic'    => $coupon->consumer->userpic,
            ],
            'cashback'       => (string)$coupon->present()->discount_consumer,
            'provision'      => (string)$coupon->present()->provision_agent,
            'validity_date'  => $coupon->validity_date->format('m.d.Y'),
            'activated_date' => $coupon->status_active_at->format('m.d.Y'),
            'needs_netto'    => false,
        ]);

        if (CouponTemplate::TYPE_PERCENT === $coupon->type_discount || CouponTemplate::TYPE_PERCENT === $coupon->type_provision) {
            $result['needs_netto'] = true;
        }

        return $this->_s($result);
    }

    /**
     * Method to calculate coupon cashback.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function calculateCashback(Request $request)
    {
        if ( !$this->user->hasRole('seller-active') ) {
            return $this->_e('Nur der Verkäufer kann den RECO.BON kalkulieren', 401);
        }

        $requestData = $this->getRequestData($request);

        $rules = [
            'code' => 'required|min:9|max:9',
        ];

        $messages = [
            'code.required' => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.min'      => 'RECO.CODE sollte 9 Symbole enthalten',
            'code.max'      => 'RECO.CODE sollte 9 Symbole enthalten',
        ];

        $validator = Validator::make($requestData, $rules, $messages);

        if ( $validator->fails() ) {
            return $this->_e($validator->errors());
        }

        $code = trim(strtoupper($requestData['code']));

        $coupon = Coupon::where('code', $code)->first();

        if (!$coupon) {
            return $this->_e('Dieser RECO.BON wurde nicht gefunden', 404);
        }

        if ($coupon->template->user_id !== $this->user->id) {
            return $this->_e('Dieser RECO.BON gehört euch nicht', 401);
        }

        switch ($coupon->type_discount) {
            case CouponTemplate::TYPE_PERCENT:
                $netto = isset($requestData['netto']) ? (float)$requestData['netto'] : 0;
                return $this->_s($netto * $coupon->discount_consumer / 100);
            case CouponTemplate::TYPE_AMOUNT:
                return $this->_s($coupon->discount_consumer);
            case CouponTemplate::TYPE_GIFT:
                return $this->_s('Geschenk');
        }
    }

    /**
     * Method to mark coupon as paid.
     *
     * @param MarkCouponAsPaid $request
     * @param Coupon $coupon
     * @return \Illuminate\Http\JsonResponse
     */
    public function markAsPaid(MarkCouponAsPaid $request, Coupon $coupon)
    {
        if ( !in_array($coupon->status, Coupon::STATUS_GROUP_CAN_BE_MARKED_PAID) ) {
            return $this->_e('Dieser RECO.BON ist schon bezahlt', 401);
        }

        (new CouponService($coupon))->setPaid();

        return $this->_s();
    }

    /**
     * Method to delete coupon
     *
     * @param DeleteCoupon $request
     * @param Coupon $coupon
     * @return \Illuminate\Http\JsonResponse
     * @throws \Exception
     */
    public function delete(DeleteCoupon $request, Coupon $coupon)
    {
        if ( Coupon::STATUS_ACTIVE !== $coupon->status ) {
            return $this->_e('Dieser RECO.BON kann nicht entfernt werden', 401);
        }

        (new CouponService($coupon))->delete();

        return $this->_s();
    }
}