<?php

namespace App;

use Digitick\Sepa\TransferFile\Factory\TransferFileFacadeFactory;
use App\Library\Billomat\BillomatClientHelper;

class CreditNote extends AbstractReceipt
{
    /**
     *  Payment of the credit note ("Auszahlung" was requested
     */
    const STATUS_PAYMENT_REQUESTED = 'payment-requested';

    const STATUS_COLORS = [
        self::STATUS_DRAFT => 'default',
        self::STATUS_OPEN=> 'info',
        self::STATUS_EXPORTED => 'warning',
        self::STATUS_PAYMENT_REQUESTED => 'primary',
        self::STATUS_COMPLETE => 'success'
    ];

    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'credit_note_id');
    }

    /**
     * Accessor to get coupon of credit note through transaction.
     *
     * @return mixed
     */
    public function getCouponAttribute()
    {
        return $this->transactions->first()->coupon;
    }

    /**
     * Transfers this Transaction to billomat as an Credit Note
     * @param type $creditnote_id
     */
    public function transferToBillomat(){

        $transaction = $this->transaction;
        $transaction->user->transferToBillomat();
        $transaction->user->fresh();
        $arr = [
            'credit-note' => [
                'client_id' => (int) $transaction->user->billomat_id,
                'credit-note-items' => [
                    'credit-note-item' => [$transaction->toBillomatItem()]
                ]
            ]
        ];

        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $ret = $billomatClient->createCreditNote($arr);

        $this->billomat_id = $ret['credit-note']['id'];

        $this->save();

        $transaction->credit_note()->associate($this)->save();

        return $this->billomat_id;
    }

    public function completeCreditNote(){
        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $arr = [
            'id' => intval( $this->billomat_id ),
            'complete' => [],
        ];


        $arr['complete']['template_id'] = (int) config('billomat.pdf.type.credit_note.templateid', null);

        $billomatClient->completeCreditNote( $arr );


        $receiptData = $billomatClient->getCreditNote( [
            'id' => intval($this->billomat_id),
        ] );

        $this->gross_amount = $receiptData['credit-note']['total_gross'];
        $this->number = $receiptData['credit-note']['credit_note_number'];
        $this->status = self::STATUS_OPEN;

        $this->save();
    }

    public function getPDF(){
        $billomatClient = BillomatClientHelper::getBillomatClientInstance();

        $ret = $billomatClient->getCreditNotePdf( [
            'id' => intval($this->billomat_id),
            'format' => 'pdf'
        ] );

        return $ret;
    }

    public function storePDF(){

        $pdf = $this->getPDF();

        $this->pdf_path = "credit_note_" . $this->billomat_id . ".pdf";

        BillomatClientHelper::storePDF('creditnotepdf', $this->pdf_path, $pdf);
    }

    public function getPDFLink(){
        return route('creditNote-downloadpdf', ['id' => $this->id]);
        //return '/creditnote/pdf/' . $this->pdf_path;
        //return Storage::disk('invoicepdf')->url( $this->pdf_path );
    }


	/**
	 * Generate credit XML, send mails (if mark exported = true)
	 *
	 * @param bool $markExportedAndSendMails
	 *
	 * @return bool|string
	 */
    public static function generateSepaCreditXml($markExportedAndSendMails = true)
    {
        $operatorAccountOwner = config('banking.accountOwner');
        $iban = config('banking.iban');
        $bic = config('banking.bic');

        $msgId = self::generateMsgId('crd');
        $directDebit = TransferFileFacadeFactory::createCustomerCredit($msgId, $operatorAccountOwner, 'pain.001.003.03');

        $receipts = self::whereStatus(self::STATUS_PAYMENT_REQUESTED)
            ->where('payment_type', 'bank')
            ->get();

        if (count($receipts) == 0) {
            return false;
        }
        /**
         * @var $t Transaction
         */
        foreach ($receipts as $receipt) {
            $paymentId = 'payment' . $receipt->id;
            $purpose = $receipt->getPurpose();
            $directDebit->addPaymentInfo($paymentId, [
                'id' => $paymentId,
                'debtorName' => $operatorAccountOwner,
                'debtorAccountIBAN' => $iban,
                'debtorAgentBIC' => $bic,
            ]);

            // Add a Single Transaction to the named payment
            $directDebit->addTransfer($paymentId, array(
                'amount' => (float)$receipt->gross_amount,
                'creditorIban' => $receipt->user->account->iban,
                'creditorName' => $receipt->user->account->iban_owner,
                'remittanceInformation' => $purpose,
            ));

            if ($markExportedAndSendMails) {
                $receipt->status = self::STATUS_EXPORTED;
                $receipt->export_run = $msgId;
                $receipt->save();

	            // mail to the receipients
	            // assumption: all the transactions have the same type (usually only one)
	            $creditType = $receipt->transactions()->first()->typeToString();
	            $amount = $receipt->gross_amount;
                $accountOwner = $receipt->user->account->iban_owner;
	            $receipt->user->sendMail('operator-exports-sepa',
		            compact('creditType', 'amount', 'accountOwner'),
		            true);
            }
        }

        $file = self::createSepaFileName($msgId);
        $xml = $directDebit->asXML();
        file_put_contents($file, $xml);
        return $file;
    }

	public function markPaidAtBillomat($date) {
		$billomatClient = BillomatClientHelper::getBillomatClientInstance();

		$arr = [
			'credit-note-payment' => [
				'credit_note_id'           => intval($this->billomat_id),
				'mark_credit_note_as_paid' => 1,
				'date'                 => $date,
				'type'                 => 'BANK_TRANSFER',
				'amount'               => $this->gross_amount,
			]
		];

		$billomatClient->createCreditNotePayment($arr);
	}

    public function getSignedAmount()
    {
        return - $this->gross_amount;
    }

}
