<?php

namespace App;

use App\Library\Utils;
use App\Services\FilterService;
use App\Services\ReferralLinkService;
use App\Services\SortingService;
use Carbon\Carbon;
use App\Presenters\CouponTemplatePresenter;
use App\Presenters\PresentsData;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Spatie\MediaLibrary\HasMedia\Interfaces\HasMediaConversions;
use Spatie\MediaLibrary\HasMedia\HasMediaTrait;


class CouponTemplate extends AbstractModel implements HasMediaConversions
{
    use PresentsData;

    use HasMediaTrait;

    public function registerMediaConversions()
    {
        $this->addMediaConversion('thumb')
            ->setManipulations(['w' => 100, 'h' => 100])
            ->performOnCollections('*');

        $this->addMediaConversion('small')
            ->setManipulations(['w' => 375, 'h' => 248])
            ->performOnCollections('*');
    }

    protected $presenter = CouponTemplatePresenter::class;

    protected $counters = [
        'visits',
        'shares',
    ];

    const STATUS_ACTIVE   = 'active';
    const STATUS_INACTIVE = 'inactive';
    const STATUS_DRAFT    = 'draft';
    const STATUS_DELETED  = 'deleted';

    const STATUS_COLORS = [
        self::STATUS_ACTIVE => 'active',
        self::STATUS_INACTIVE => 'inactive',
    ];

    const TYPE_PERCENT = 'percent';
    const TYPE_AMOUNT  = 'amount';
    const TYPE_GIFT    = 'gift';
    const TYPE_NONE    = 'none';

    const TYPE_DEFAULT = self::TYPE_PERCENT;

    const DISCOUNT_TYPES = [
        self::TYPE_PERCENT,
        self::TYPE_AMOUNT,
        self::TYPE_GIFT,
    ];

    const PROVISION_TYPES = [
        self::TYPE_NONE,
        self::TYPE_PERCENT,
        self::TYPE_AMOUNT,
        self::TYPE_GIFT,
    ];

    protected $guarded = [];

    protected $dates = ['expiration_date'];

    protected $appends = ['coupon_count'];

    public function category()
    {
        return $this->hasOne('\App\Category', 'id', 'category_id');
    }

    public function user()
    {
        return $this->hasOne('\App\User', 'id', 'user_id');
    }

    public function seller()
    {
        return $this->user();
    }

    public function coupons()
    {
        return $this->hasMany('\App\Coupon', 'coupon_template_id', 'id');
    }

    public function couponsRedeemed()
    {
        return $this->hasMany('\App\Coupon', 'coupon_template_id', 'id')
            ->whereIn('status', [Coupon::STATUS_PAID, Coupon::STATUS_DONE]);
    }

    public function provisionGift()
    {
        return $this->hasOne('\App\Gift', 'id', 'provision_gift_id');
    }

    public function discountGift()
    {
        return $this->hasOne('\App\Gift', 'id', 'discount_gift_id');
    }

    public function favoriters()
    {
        return $this->belongsToMany('App\User', 'favorites', 'deal_id', 'user_id');
    }

    public function places()
    {
        return $this->belongsToMany('App\Place', 'deal_place', 'deal_id', 'place_id');
    }

    public function ratings()
    {
        return $this->hasMany('\App\Rating', 'rateable_id', 'id')
            ->where('rateable_type', 'App\CouponTemplate');
    }

    public function activities()
    {
        return $this->morphMany('\App\Activity', 'model');
    }

    /**
     * Scope a query to only include available deals.
     *
     * @param $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAvailable($query)
    {
        return $query
            ->with([/*'provisionGift', 'discountGift', */'seller.account', 'category', 'media'])
            ->withCount('coupons')
            ->where('expiration_date', '>=', Carbon::today())
            ->where('status',          '=',  self::STATUS_ACTIVE)
            ->having('quota',          '>',  DB::raw('coupons_count'))
            ->whereHas('seller', function ($query) {
                $query->where('status', User::STATUS_ACTIVE);
            });
    }

    /**
     * Scope a query to get top deals (most shared and visited).
     *
     * @param $query
     * @return mixed
     */
    public function scopeTop($query)
    {
        return $query
            ->orderBy('shares', 'DESC')
            ->orderBy('visits', 'DESC');
    }

    /**
     * Scope a query to only include draft deals.
     *
     * @param $query
     * @return mixed
     */
    public function scopeDraft($query)
    {
        return $query->where('status', self::STATUS_DRAFT);
    }

    /**
     * Scope a query to only include deals older than interval in minutes.
     *
     * @param $query
     * @param $minutes
     * @return mixed
     */
    public function scopeOlderThan($query, $minutes)
    {
        return $query->where('updated_at', '<=', Carbon::now()->subMinutes($minutes)->toDateTimeString());
    }

    /**
     * @param $status array/status
     * @return mixed
     */
    public function couponsWithStatus($status)
    {
        if (is_array($status)) {
            return $this->coupons()
                ->whereIn('status', $status);
        }
        return $this->coupons()
            ->where('status', '=', $status);
    }

    public function countCouponsWithStatus($status)
    {
        return $this->couponsWithStatus($status)->count();
    }

    public function couponsForAgentAccount(Account $agentAccount)
    {
        return $this->coupons()->whereIn('agent_user_id', $agentAccount->getUserIds());
    }

    public function getDisplayName()
    {
        return $this->title;
    }

    /**
     * @return bool
     * true if limit is reached
     * false if limit is not reached
     */
    public function limitReached()
    {
        $couponsCount = $this->coupons()->count();
        if($couponsCount < $this->quota){
            return false;
        }
        return true;
    }

    public function isExpired()
    {
        return $this->expiration_date->lt(Carbon::today());
    }

    public function isShareable()
    {
        return !$this->isExpired() && !$this->limitReached();
    }

    public function getSellerMaxCashback()
    {
        return $this->orderBy('discount_consumer','DESC')->first()->discount_consumer;
    }

    public function getSellerMaxRecoCash()
    {
        return $this->orderBy('provision_agent','DESC')->first();
    }

    public function isActive()
    {
        return self::STATUS_ACTIVE === $this->status;
    }

    /**
     * Method to check whether standalone deal is favorited by user
     *
     * @return bool
     */
    public function isFavorited()
    {
        $favs = session('favs', collect([]));

        return $favs->contains($this->id);
    }

    /**
     * Coupon count accessor.
     *
     * @return int
     */
    public function getCouponCountAttribute()
    {
        return $this->quota - $this->coupons->count();
    }

    /**
     * Method to get available deals (filtered if it necessary).
     *
     * @param bool $useFilters
     * @return mixed
     */
    public static function getAvailableDeals($useFilters = true)
    {
        $query = self::available();

        if ($useFilters) {
            $query = (new FilterService($query))->apply();
        }

        $query = (new SortingService($query))->apply();

        return $query->get();
    }

    /**
     * Referral links accessor.
     *
     * @return object
     */
    public function getLinksAttribute()
    {
        return (new ReferralLinkService(Auth::user(), null, $this))->getAllLinks();
    }

    /**
     * Gallery accessor.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getGalleryAttribute()
    {
        return $this->media
            ->sortBy('order_column')
            ->where('collection_name', 'images');
    }

    /**
     * Video accessor.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getVideoAttribute()
    {
        return $this->media
            ->sortBy('order_column')
            ->where('collection_name', 'videos');
    }

    /**
     * Main image accessor (gallery first image or img)
     *
     * @return mixed
     */
    public function getMainImageAttribute()
    {
        $images = $this->getMedia('images');
        if ( $images->count() ) {
            return $images[0]->getUrl('small');
        }

        // backward compatibility
        return route('coupon.get-image', $this->img);
    }

    /**
     * Search string accessor.
     *
     * @return bool|mixed|null|string|string[]
     */
    public function getSearchStringAttribute()
    {
        return Utils::prepareSearchString($this->title);
    }

    /**
     * Method to get deal rating by user.
     *
     * @param null $user
     * @return bool
     */
    public function ratingByUser($user = null)
    {
        if (null === $user) {
            $user = Auth::user();
        }

        return $this->ratings
            ->where('user_id', $user->id)
            ->first();
    }

    /**
     * Check whether deal is in draft status.
     *
     * @return bool
     */
    public function isDraft()
    {
        return self::STATUS_DRAFT === $this->status;
    }

    /**
     * Check whether deal is deleted.
     *
     * @return bool
     */
    public function isDeleted()
    {
        return self::STATUS_DELETED === $this->status;
    }
}
