<?php

namespace App;

use App\Library\Facades\Format;
use App\Services\BarCodeService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use App\Presenters\CouponPresenter;
use App\Presenters\PresentsData;

class Coupon extends AbstractModel
{
    use PresentsData;

    protected $presenter = CouponPresenter::class;

    protected $fillable = [
        'coupon_template_id',
        'consumer_user_id',
        'agent_user_id',
        'title',
        'expiration_date',
        'description',
        'provision_agent',
        'discount_consumer',
        'status',
        'code',
        'selection',
        'selection_group',
        'emailtext',
        'smstext',
        'validity_date',
        'type_provision',
        'type_discount',
        'provision_gift_id',
        'discount_gift_id',
    ];

    protected $dates = [
        'validity_date',
        'status_active_at',
    ];

    /** @startuml
     * [*] --> pending: Coupon is created
     * pending->active: Customer accepts coupon
     * active->redeemed: Customer makes a purchase at seller and provides code
     * redeemed->paid: Seller confirms the payment of the purchase
     * paid->done: Transactions and invoices are generated
     * done->[*]
     * @enduml
     *
     */
    const STATUS_PENDING = 'pending';
    const STATUS_ACTIVE = 'active';
    const STATUS_REDEEMED = 'redeemed';
    /**
     * Consumer claims that s/he payed the coupon (but not yet confirmed by the seller)
     */
    const STATUS_PAYMENT_ASSERTED = 'payment_asserted';
    /**
     * Seller marked the coupon as paid
     */
    const STATUS_PAID = 'paid';
    const STATUS_DONE = 'done';

    const STATUS_COLORS = [
        self::STATUS_PENDING => 'warning',
        self::STATUS_ACTIVE=> 'info',
        self::STATUS_REDEEMED => 'warning',
        self::STATUS_PAID => 'success',
        self::STATUS_DONE => 'success',
        self::STATUS_PAYMENT_ASSERTED => 'success',
    ];

    const STATUS_TITLES = [
        self::STATUS_PENDING          => 'Steht aus',
        self::STATUS_ACTIVE           => 'Aktiv',
        'recommended'                 => 'Empfohlen',
        self::STATUS_REDEEMED         => 'Eingelöst',
        self::STATUS_PAID             => 'Bezahlt',
        self::STATUS_DONE             => 'Abgeschlossen',
        self::STATUS_PAYMENT_ASSERTED => 'Als bezahlt markiert',
    ];

    // FIXME: what is this for? -> document
    const STATUS_GROUP_ACTIVE = [
        self::STATUS_ACTIVE,
        self::STATUS_PAYMENT_ASSERTED,
        self::STATUS_REDEEMED,
        self::STATUS_DONE,
        self::STATUS_PAID
    ];

    // FIXME: same like finished??
    const STATUS_GROUP_FINISHED_STAMP = [
        self::STATUS_REDEEMED,
        self::STATUS_DONE,
    ];

	/**
	 * Everything after redeemed
	 * We might later have a status canceled,
	 * which should not be in this group
	 */
    const STATUS_GROUP_FINISHED = [
        self::STATUS_REDEEMED,
	    self::STATUS_PAYMENT_ASSERTED,
        self::STATUS_DONE,
        self::STATUS_PAID
    ];

    /**
     * Status from which the coupon can be marked paid
     */
    const STATUS_GROUP_CAN_BE_MARKED_PAID = [
        self::STATUS_REDEEMED,
        self::STATUS_PAYMENT_ASSERTED
    ];

	/**
	 * Coupon is paid (or done, which can be reached only if payed)
	 */
    const STATUS_GROUP_IS_PAID = [
    	self::STATUS_PAID,
        self::STATUS_DONE,
    ];

    public function template()
    {
        return $this->belongsTo('\App\CouponTemplate','coupon_template_id', 'id');
    }

    public function deal()
    {
        return $this->belongsTo('\App\CouponTemplate','coupon_template_id', 'id');
    }

    public function consumer()
    {
        return $this->belongsTo('\App\User', 'consumer_user_id', 'id');
    }

    public function agent()
    {
        return $this->belongsTo('\App\User', 'agent_user_id', 'id');
    }

    public function redeemer()
    {
        return $this->belongsTo('\App\User', 'redeemer_id', 'id');
    }

    public function transactions() {
        return $this->hasMany('App\Transaction');
    }

    public function creditNoteTransactions(){
        return $this->transactions()->whereIn('type', Transaction::TYPE_CREDIT);
    }

    public function paymentTransactions(){
        return $this->transactions()->whereIn('type', Transaction::TYPE_PAYMENT);
    }

    public function provisionGift()
    {
        return $this->hasOne('\App\Gift', 'id', 'provision_gift_id');
    }

    public function discountGift()
    {
        return $this->hasOne('\App\Gift', 'id', 'discount_gift_id');
    }

    public function activities()
    {
        return $this->morphMany('\App\Activity', 'model');
    }

    public function getTransactionAmountFormatted($type)
    {
        $subTransaction = $this->getTransactionByType($type);
        if (!$subTransaction) {
            return '';
        }
        return Format::currency($subTransaction->amount);
    }

    public function getTransactionStatusText($type)
    {
        $subTransaction = $this->getTransactionByType($type);
        if (!$subTransaction) {
            return '';
        }
        return $subTransaction->getStatusText();
    }

    public function getTransactionPDFLink($type)
    {
        $subTransaction = $this->getTransactionByType($type);
        if (!$subTransaction) {
            return '';
        }
        if (($pdfLink = $subTransaction->getPDFLink()) == '') {
            return '';
        }
        return new HTMLString(
            '<a href="' . e($pdfLink) . '" target="_blank"><i class="fa fa-file-pdf-o"></i></a> '
        );
    }


    public function getReceiptStatusText($type)
    {
        $subTransaction = $this->getTransactionByType($type);
        $receipt = $subTransaction->getReceipt();
        if (!$receipt) {
            return '';
        }
        return $receipt->getStatusText();
    }


    public function invoice(){
        return $this->hasOne('App\Invoice');
    }

    public function createInvoice() {

        //Create Invoice
        $invoice = new Invoice();
        $invoice->coupon_id = $this->id;

        $invoice->transferCouponInvoiceToBillomat();
        $invoice->user_id = $this->template->user_id; // seller

        $invoice->completeInvoice();

        $invoice->storePDF();

        $invoice->save();
    }

    public function getCodeFormatted()
    {
        return Format::recoCode($this->code);

    }
    public function getPDFLink(){
        if($this->invoice instanceof Invoice) {
            return $this->invoice->getPDFLink();
        }
        return false;
    }

    /**
     * Method to redeem coupon.
     *
     * @param $net
     */
    public function redeem($net) {
        $this->status      = Coupon::STATUS_REDEEMED;
        $this->netto       = $net;
        $this->redeemer_id = Auth::user()->id;
        $this->save();
    }

    /**
     * All coupons from the current selection group
     */
    public function fullSelectionGroup()
    {
        return $this->hasMany(self::CLASS, 'selection_group', 'selection_group')
            ->where('selection_group', '>', 0);
    }

    /**
     * Get maxiumi cashback for a selection coupon / group
     * @return int|mixed
     */
    public function getMaxCashback()
    {
        /**
         * @var $coupon Coupon
         */
        $max = 0;

        foreach($this->fullSelectionGroup as $coupon) {
            $discount = $coupon->template->discount_consumer;
            if ($discount > $max) {
                $max = $discount;
            }
        }

        return max($max, $this->discount_consumer);
    }

    public function calculateConsumerDiscount()
    {
        // TODO if gift
        if ( CouponTemplate::TYPE_PERCENT === $this->type_discount ) {
            if (!$this->netto) {
                return null;
            }
            return $this->netto * $this->discount_consumer / 100;
        } else {
            return $this->discount_consumer;
        }
    }

    public function calculateAgentProvision()
    {
        // TODO if gift
        // TODO if none
        if ( CouponTemplate::TYPE_PERCENT === $this->type_provision ) {
            if (!$this->netto) {
                return null;
            }
            return $this->netto * $this->provision_agent / 100;
        } else {
            return $this->provision_agent;
        }
    }

    /**
     * @param $type
     * @return mixed
     */
    public function getTransactionByType($type)
    {
        $subTransaction = $this->transactions()
            ->whereType($type)->first();
        return $subTransaction;
    }

    /**
     * Execute count on an eloquent query where selection coupons are only counted once
     *
     * @param $query
     * @return mixed
     */
    public static function countMultiBonOnlyOnce($query)
    {
        return $query->count(DB::raw('DISTINCT IF(selection_group = 0, -coupons.id, coupons.selection_group)'));
    }

    /**
     * Main image accessor (gallery first image or img)
     *
     * @return mixed
     */
    public function getMainImageAttribute()
    {
        $images = $this->template->getMedia('images');

        if ( $images->count() ) {
            return $images[0]->getUrl('small');
        }

        // backward compatibility
        return route('coupon.get-image', $this->template->img);
    }

    /**
     * Barcode image accessor.
     *
     * @return mixed
     */
    public function getBarCodeAttribute()
    {
        return (new BarCodeService($this->code))->getBarcode();
    }

    /**
     * QR-code image accessor.
     *
     * @return mixed
     */
    public function getQRCodeAttribute()
    {
        return (new BarCodeService($this->code))->getQRCode();
    }
}
