<?php

namespace App;

use App\Library\Utils;
use App\Services\FilterService;
use App\Services\SortingService;
use Baum\Node;
use Illuminate\Support\Facades\DB;

class Category extends Node
{
    use HasMedia;

    protected $table = 'categories';

    protected $mediaAttributes = ['icon'];

    public function couponTemplates()
    {
        return $this->hasMany('\App\CouponTemplate');
    }

    /**
     * Method to get category tree with (or without) counted coupon templates belongs to each node.
     *
     * @param bool $countCouponTemplates Need to count
     * @param bool $aggregateCount Need to aggregate (count for all subtree of every node)
     * @param bool $addChain
     * @return mixed
     */
    public static function getTree($countCouponTemplates = false, $aggregateCount = false, $addChain = false)
    {
        $categories = Category::all()->keyBy('id');

        if ($countCouponTemplates) {
            $couponTemplateCounts = self::getCouponTemplateCounts();

            foreach ($couponTemplateCounts as $k => $v) {
                if ( !$aggregateCount && !$categories->find($k)->isLeaf() ) {
                    continue;
                }
                $categories->find($k)->couponTemplatesCount = $v->cnt;
            }
        }

        if ($addChain) {
            foreach ($categories as $category) {
                $chain = collect([]);
                $parent = isset($categories[$category->parent_id]) ? $categories[$category->parent_id] : null;
                while ($parent) {
                    $chain->prepend($parent);
                    $parent = isset($categories[$parent->parent_id]) ? $categories[$parent->parent_id] : null;
                }
                $category->chain = $chain;
            }
        }

        return $categories->toHierarchy();
    }

    /**
     * Method to update category URL after parent change.
     */
    public function updateURL()
    {
        $slugs = [$this->slug];

        if ($this->parent_id) {
            $parent = Category::find($this->parent_id);
            $slugs[] = $parent->slug;

            if ($parent->parent_id) {
                $parent = Category::find($parent->parent_id);
                $slugs[] = $parent->slug;
            }
        }

        $slugs = array_reverse($slugs);
        $url = implode('/', $slugs);

        $this->url = $url;
        $this->save();
    }

    /**
     * Method to get top level categories.
     *
     * @return mixed
     */
    public static function getTop()
    {
        return Category::where('depth', 0)->orderBy('lft')->get();
    }

    /**
     * Method to get collection of coupon template counts by categories.
     *
     * @param bool $onlyActiveDeals
     * @return \Illuminate\Support\Collection
     */
    private static function getCouponTemplateCounts($onlyActiveDeals = false)
    {
        $activeDealsQuery = $onlyActiveDeals
            ? 'AND coupon_templates.status = "active"'
            : '';

        $sql = '
            SELECT parent.id, COUNT(coupon_templates.id) AS cnt
            FROM categories AS node, categories AS parent, coupon_templates
            WHERE node.lft BETWEEN parent.lft AND parent.rgt
            AND node.id = coupon_templates.category_id
            ' . $activeDealsQuery . '
            GROUP BY parent.id
        ';

        return collect(DB::select($sql))->keyBy('id');
    }

    /**
     * Method to check if category has any coupon templates.
     *
     * @return bool
     */
    public function hasCouponTemplates()
    {
        $couponTemplateCounts = self::getCouponTemplateCounts();

        return isset($couponTemplateCounts[$this->id]) && $couponTemplateCounts[$this->id]->cnt;
    }

    /**
     * Method to get available deals by category.
     *
     * @param bool $withChildren
     * @param bool $useFilters
     * @return mixed
     */
    public function getAvailableDeals($withChildren = true, $useFilters = true)
    {
        if ($withChildren) {
            $categories = Category::whereBetween('lft', [$this->lft, $this->rgt])->pluck('id');
        } else {
            $categories = [$this->id];
        }

        $deals = CouponTemplate::available()
            ->whereIn('category_id', $categories);

        if ($useFilters) {
            $deals = (new FilterService($deals))->apply();
        }

        $deals = (new SortingService($deals))->apply();

        $deals = $deals->get();

        return $deals;
    }

    /**
     * Method to get category breadcrumbs chain.
     *
     * @return array
     */
    public function getBreadcrumbs()
    {
        $crumbs = [];

        $items = $this->getAncestors();

        foreach ($items as $item) {
            $crumbs[] = (object)[
                'title' => $item->name,
                'url'   => route('category.index', ['hierarchy' => $item->url]),
            ];
        }

        $crumbs[] = (object)['title' => $this->name];

        return $crumbs;
    }

    /**
     * Parent names chained string accessor.
     *
     * @return string
     */
    public function getParentChainAttribute()
    {
        $chain = [];

        $items = $this->getAncestors();

        foreach ($items as $item) {
            $chain[] = $item->name;
        }

        return implode(' / ', $chain);
    }

    /**
     * Search string accessor.
     *
     * @return bool|mixed|null|string|string[]
     */
    public function getSearchStringAttribute()
    {
        return Utils::prepareSearchString($this->name);
    }

    /**
     * Get categories by popularity (available deals count).
     *
     * @param int $limit
     * @return mixed
     */
    public static function getPopular($limit = 10)
    {
        $couponTemplateCounts = self::getCouponTemplateCounts(true);
        $categories = self::getTop();

        foreach ($categories as $category) {
            $category->cnt = $couponTemplateCounts[$category->id]->cnt ?? 0;
        }

        return $categories->sortByDesc('cnt')->slice(0, $limit);
    }
}
