<?php

namespace App;

use App\Events\CouponFinishedClientTransactions;
use App\Services\AqBankingService;
use AqBanking\HbciVersion;
use AqBanking\PinFile\PinFileCreator;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

use AqBanking\Account as AqAccount;
use AqBanking\Bank;
use AqBanking\BankCode;
use AqBanking\Command\AddUserCommand;
use AqBanking\Command\GetSysIDCommand;
use AqBanking\Command\RenderContextFileToXMLCommand;
use AqBanking\Command\RequestCommand;
use AqBanking\ContextFile;
use AqBanking\ContextXmlRenderer;
use AqBanking\PinFile\PinFile;
use AqBanking\User as AqUser;
use AqBanking\Transaction as AqTransaction;
use Illuminate\Support\Facades\Log;

use App\Events\CouponFinishedSellerTransactions;

class BankTransaction extends AbstractModel
{
    protected $dates = [
        'created_at',
        'updated_at',
        'date',
        'valuta_date',
    ];

    public $incrementing = false;

    public function credit_notes()
    {
        return $this->hasMany(CreditNote::class);
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * NOTE: We return only the first receipt even it would be theoretically possible
     * to have multiple receipts linked to the same bank_transaction
     *
     * @return Model|null|App\BankTransaction|static
     */
    public function getReceipt()
    {
        if ($this->credit_notes()->count() >0 ) {
            return $this->credit_notes->first();
        } else
            if ($this->invoices()->count() >0 ) {
                return $this->invoices->first();
        }

        return null;
    }

    /**
     * Get (and auto-create) storage path
     * @return string
     */
    public static function getAqStoragePath()
    {
        $path = storage_path('app/aqbanking');
        if (!file_exists($path)) {
            mkdir($path, 0770, true);
        }
        return $path;
    }

    public static function getAqUser()
    {
        $bankCodeString = config('banking.bankCodeString');
        $bankUrl = config('banking.aqBankUrl');
        $bankCode = new BankCode($bankCodeString);
        $bank = new Bank($bankCode, $bankUrl, new HbciVersion('300'));
        $user = new AqUser(config('banking.userId'), config('banking.accountOwner'), $bank);
        return $user;
    }

    /**
     * @return AqAccount
     */
    public static function getAqAccount()
    {
        $bankCodeString = config('banking.bankCodeString');
        $accountNumber = config('banking.accountNumber');
        $bankCode = new BankCode($bankCodeString);

        $account = new AqAccount($bankCode, $accountNumber);
        return $account;
    }

    /**
     * @return ContextFile
     */
    public static function getAqContextFile()
    {
        $contextFile = new ContextFile(self::getAqStoragePath() . DIRECTORY_SEPARATOR . 'aqBanking.ctx');
        return $contextFile;
    }

    public static function addUser()
    {
        $addUser = new AddUserCommand();
        $addUser->execute(self::getAqUser());
    }

    public static function createPinFile()
    {
        $pfc = new PinFileCreator(self::getAqStoragePath());
        $pin = config('banking.pin');
        if (empty($pfc)) {
            throw new \Exception("Banking PIN not configured");
        }
        $pfc->createFile($pin, self::getAqUser());
    }

    public static function executeGetSysId()
    {
        $user = self::getAqUser();
        $pinFile = new PinFile(self::getAqStoragePath(), $user);

        $getSysId = new GetSysIDCommand();
        $getSysId->execute($user, $pinFile);
    }

    /**
     * Create user, PIN file etc.
     * If the user already exists, no PIN file or sysid calls are made
     */
    public static function initializeAqbanking()
    {
        try {
            BankTransaction::addUser();
            BankTransaction::createPinFile();
            BankTransaction::executeGetSysId();
        } catch (\AqBanking\Command\AddUserCommand\UserAlreadyExistsException $e) {
            // it's okay if the user already exists
        }
    }

    public static function runAqRequest()
    {
        self::initializeAqbanking();
        $account = self::getAqAccount();
        $user = self::getAqUser();
        $pinFile = new PinFile(self::getAqStoragePath(), $user);
        $contextFile = self::getAqContextFile();
        $sut = new RequestCommand($account, $contextFile, $pinFile);
        $sut->execute();
        $render = new RenderContextFileToXMLCommand();
        $dom = $render->execute($contextFile);
        $result = new ContextXmlRenderer($dom);
        return $result;
    }

    public static function getBalance()
    {
        $result = self::runAqRequest();
        return $result->getBalance()->getAmount() . ' ' . $result->getBalance()->getCurrency();
    }

    /**
     * @return AqTransaction[]
     * @throws \AqBanking\Command\ShellCommandExecutor\DefectiveResultException
     */
    public static function getTransactions()
    {
        $dateFrom = Carbon::now()->subdays(1);
        return (new AqBankingService())->getTransactions($dateFrom);
    }

    /**
     * Format key
     * @param $date Date object
     * @param $increment Increment of the data
     * @return int Key
     */
    public static function formatKey($date, $increment)
    {
        if ($date instanceof Carbon) {
            $date = $date->format('Ymd') ;
        }

        return $date . sprintf('%05d', $increment);
    }

    /**
     * Generate a unique ID per transaction
     * We assume, that all transactions of a day are always present in the fetched transactions
     * (after a while, very old transactions might not appear)
     * So we generate a continues key per day, combine it with the date and have a globally unique key
     *
     * @return AqTransaction[]
     */
    public static function getKeyedTransactions()
    {
        $transactions = self::getTransactions();
        $result = [];
        $dateCounters = array();
        foreach ($transactions as $transaction) {
            $date = $transaction->getDate()->format('Ymd');
            if (!isset($dateCounters[$date])) {
                $dateCounters[$date] = 0;
            }
            $dateCounters[$date]++;
            $uniqId = self::formatKey($date, $dateCounters[$date]);
            $result[$uniqId] = $transaction;
        }
        return $result;
    }

    public static function importTransactions()
    {
        $transactions = self::getKeyedTransactions();

        foreach ($transactions as $id => $aqTransaction) {
            if (self::find($id)) {
                continue;
            }

            $transaction = new self();
            $transaction->id = $id;
            $transaction->date = $aqTransaction->getDate();
            $transaction->valuta_date = $aqTransaction->getValutaDate();
            $transaction->remote_account_holder_name = $aqTransaction->getRemoteAccount()->getAccountHolderName();
            $transaction->remote_account_number = $aqTransaction->getRemoteAccount()->getAccountNumber();
            $transaction->remote_bank_code = $aqTransaction->getRemoteAccount()->getBankCode()->getString();
            $transaction->local_account_holder_name = $aqTransaction->getLocalAccount()->getAccountHolderName();
            $transaction->local_account_number = $aqTransaction->getLocalAccount()->getAccountNumber();
            $transaction->local_bank_code = $aqTransaction->getLocalAccount()->getBankCode()->getString();
            $transaction->amount = $aqTransaction->getValue()->getAmount() / 100;
            $transaction->currency = $aqTransaction->getValue()->getCurrency()->getName();
            $transaction->purpose = $aqTransaction->getPurpose();

            $transaction->save();

            $transaction->parseReceipt();
        }
    }


    public function parseReceipt()
    {
        $item = $this->purpose;

        //check if the item is retoure
        if(preg_match('/Retoure/', $item)){
            preg_match('/(RE)[0-9]+/', $item, $match);
            //get the receipt from database
            $receipt = Invoice::whereNumber($match[0])->first();

            if($receipt == null){
                Log::warning('Matched Format, but not found in database', ['bankTransaction' => $this]);
            }
            //assign status booked_back to the receipt
            $receipt->status = AbstractReceipt::STATUS_BOOKED_BACK;
            $receipt->save();

            $seller = User::where('id', '=', $receipt->user_id)->first();

            $operators = User::whereHas('roles', function ($q) {
                $q->where('name', 'operator');
            })->get();

            foreach ($operators as $operator) {
                $operator->sendMail('operator-notification-booked-back', compact('receipt', 'seller'), true);
            }

            // TODO cancel transactions ???

            return;
        }

        if (preg_match('/(RE|GS)[0-9]+/', $item, $matches)) {
            $receiptNumber = $matches[0];
            if (preg_match('/(RE)/', $receiptNumber)) {
                $receipt = Invoice::whereNumber($receiptNumber)->first();
                if ($receipt == null) {
                    Log::warning('Matched RE format, but not found in database',
                        ['bankTransaction' => $this]);
                    return;
                }
            } else if (preg_match('/(GS)/', $receiptNumber)) {
                $receipt = CreditNote::whereNumber($receiptNumber)->first();
                if ($receipt == null) {
                    Log::warning('Matched GS format, but not found in database',
                        ['bankTransaction' => $this]);
                    return;
                }
            } else {
                return;
            }

            $receiptAmount = $receipt->getSignedAmount();

            if ($receiptAmount != $this->amount) {
                Log::warning('Amount not matching',
                    ['receipt' => $receipt, 'bankTransaction' => $this]);
                return;
            }

            if ($receipt->bank_transaction_id) {
                Log::warning('Receipt already assigned',
                    ['receipt' => $receipt, 'bankTransaction' => $this]);
                return;
            }

            $this->assignBankTransaction($receipt);
        }
    }

    public function assignBankTransaction($receipt)
    {
        $receipt->status = AbstractReceipt::STATUS_COMPLETE;
        //FIXME: transfer paid status to billomat
        $receipt->bank_transaction_id = $this->id;
        $receipt->save();
        $receipt->markPaidAtBillomat($this->valuta_date);

        foreach ($receipt->transactions as $transaction) {
            $transaction->status = Transaction::STATUS_PAID;
            $transaction->save();
        }

        // we have successfully got money from seller
        if ($receipt instanceof Invoice) {
            event(new CouponFinishedSellerTransactions($receipt->coupon));
        }

        // we have successfully sent money to client
        if ($receipt instanceof CreditNote) {
            event(new CouponFinishedClientTransactions($receipt->coupon));
        }
    }

    /**
     * Create fake bank transaction from receipt and parse the receipt
     * @param AbstractReceipt $receipt
     */
    public static function createFakeFromReceipt(AbstractReceipt $receipt, $parse = true, $purpose = '')
    {
        $bankTransaction = new BankTransaction();

        $date = \Carbon\Carbon::today();
        $increment = 0;

        do {
            $increment++;
            $key = BankTransaction::formatKey($date, $increment);
        } while (!is_null(BankTransaction::find($key)));

        $bankTransaction->id = $key;
        $bankTransaction->date = $date;
        $bankTransaction->valuta_date = $date;
        $bankTransaction->remote_account_holder_name = $receipt->user->getDisplayName();
        $bankTransaction->remote_account_number = '0815';
        $bankTransaction->remote_bank_code = '12345678';
        $bankTransaction->local_account_holder_name = 'RECO MA';
        $bankTransaction->local_account_number =  '123456789';
        $bankTransaction->local_bank_code = '23456789';
        $bankTransaction->amount = $receipt->getSignedAmount();
        $bankTransaction->currency = 'EUR';
        $bankTransaction->purpose = $purpose . ' ' . $receipt->number;
        $bankTransaction->save();

        if ($parse) {
            $bankTransaction->parseReceipt();
        }
    }

}