<?php

namespace App;

use App\Http\Controllers\Onboarding\OnboardingAgentController;
use App\Library\AccountStatusScope;
use App\Http\Controllers\Onboarding\OnboardingSellerController;
use Carbon\Carbon;
use Digitick\Sepa\Exception\Exception;
use Equi\Opengeodb\Models\GeodbMaster;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\HtmlString;
use willvincent\Rateable\Rateable;
use PDF;

class Account extends AbstractModel
{
    use Rateable;

    use HasMedia;

    protected $fillable = [
        'iban',
        'iban_owner',
        'company',
        'street',
        'housenumber',
        'zip',
        'city',
        'tax_id',
        'payment',
        'status',
        'upload_portrait',
        'description',
        'all_connections',
        'billed_until',
        'recruiter_user_id',
        'picture',
        'worktimes',
    ];

    protected $mediaAttributes = ['picture'];

    protected $appends = ['logo'];

    protected static function boot()
    {
        parent::boot();
        static::addGlobalScope(new AccountStatusScope);
    }

    protected $virtual_attributes = ['displayName'];

    protected $casts = [
        'billed_until' => 'date',
        'received_seller_role_at' => 'date',
    ];

    public function users()
    {
        return $this->hasMany('\App\User', 'account_id', 'id');
    }

    public function ratings()
    {
        return $this->hasMany('App\Rating', 'rated_account_id', 'id');
    }

    public function recruiterUser()
    {
        return $this->belongsTo(User::class, 'recruiter_user_id');
    }
    /**
     * Get user IDs for this account
     * @return mixed
     */
    public function getUserIds()
    {
        return $this->users()->getQuery()->select('id')->pluck('id');
    }


    public function relatedAccounts()
    {
        return $this->belongsToMany('\App\Account', 'account_connections', 'account_id', 'related_account_id');
    }

    public function relatedSellerAccounts()
    {
        return $this->relatedAccounts()->wherePivot('type', 'agent');
    }

    public function relatedFromAccount()
    {
        return $this->belongsToMany('\App\Account', 'account_connections', 'related_account_id', 'account_id');
    }

    public function relatedAgentAccounts()
    {
        return $this->relatedFromAccount()->wherePivot('type', 'agent')->distinct();
    }

    public function relatedAccountConnectionsViaNewsletter(){
        return $this->relatedAccounts()->wherePivot('type',AccountConnection::NEWSLETTER_REQUEST)->wherePivot('status',AccountConnection::STATUS_ACTIVE);
    }

    /**
     * Connected sellers and those who allow all to sell
     * @return mixed
     */
    public function availableSellerAccountsQuery()
    {
        $query = $this->relatedSellerAccounts()->getQuery()
            ->select(['accounts.*', 'account_connections.related_account_id as related_account_id']);
        return $query;
    }

    /**
     * Connected agents
     * @return mixed
     */
    public function availableAgentAccountsQuery()
    {
        $query = $this->relatedAgentAccounts()->getQuery()
            ->select(['accounts.*', 'account_connections.related_account_id']);
        return $query;
    }

    public function sellersWhereICanApplyToQuery($includeAlreadyApplied = true)
    {
        $available = $this->availableSellerAccountsQuery()->get('id')->pluck('id');
        $sellers = Account::allSellerAccounts()->pluck('id');
        return $this->query()->whereIn('accounts.id', $sellers)->whereNotIn('accounts.id', $available);
    }

    public function agentsWhereICanApplyToQuery()
    {
        $available = $this->availableAgentAccountsQuery()->get('id')->pluck('id');
        $agents = Account::allAgentAccounts()->pluck('id');
        return $this->query()->whereIn('accounts.id', $agents)->whereNotIn('accounts.id', $available);
    }

    /**
     * Get all users (unique) that have a coupon of this seller account
     * @return Collection
     * @throws \Exception
     */
    public function getUsersWithCouponsFromSellersOfAgent()
    {
        $userIdsOfAccount = $this->users->pluck('id')->toArray();
        $couponTemplates = CouponTemplate::whereIn('user_id', $userIdsOfAccount)->get()->pluck('id')->toArray();
        $usersWithCoupons = new Collection();
        foreach ($couponTemplates as $couponTemplate) {
            $coupons = $couponTemplate->coupons;
            $usersWithCoupons = $usersWithCoupons->union($coupons->pluck('consumer')->keyBy('id'));
        }
        return $usersWithCoupons;
    }

    public function couponTemplates()
    {
        return $this->hasManyThrough('\App\CouponTemplate', '\App\User', 'account_id', 'user_id', 'id');
    }

    public function getDisplayName()
    {
        if ($this->company != '') {
            return $this->company;
        } else {
            return $this->users()->first()->getDisplayName();
        }
    }

    public function getPortraitUrl($role = 'seller')
    {
        if ($role == User::SELLER && $this->upload_portrait != '') {
            return route('user-portrait', $this->upload_portrait);
        }

        if ($role != User::SELLER && $this->users->first()->upload_portrait != '') {
            return route('user-portrait', $this->users->first()->upload_portrait);
        }

        if ($role == User::SELLER) {
            return url('/img/logo_company.png');
        }

        return url('/img/default.jpg');
    }

    public static function allOf($type)
    {
        // FIXME: do it on DB level
        $method = 'all' . ucfirst(str_plural($type));
        $result = new Collection();
        foreach (User::$method()->get() as $user) {
            $result[] = $user->account;
        }
        return $result;

    }

    public static function allSellerAccounts()
    {
        return self::allOf(User::SELLER);
    }

    public static function allAgentAccounts()
    {
        return self::allOf(User::AGENT);
    }

    public static function allConsumerAccounts()
    {
        return self::allOf(User::CONSUMER);
    }

    public function getLink()
    {
        return route('show-seller-account', $this->id);
    }

    public function getLinkedDisplayName()
    {
        return new HtmlString('<a class="" href="' . $this->getLink() . '">' . e($this->getDisplayName()) . '</a>');
    }

    public function getIconLink()
    {
        return new HtmlString('<a class="btn btn-white" class="" href="' . e($this->getLink()) . '">' . '<i class="fa fa-user"></i></a>');
    }

    public function invitations()
    {
        return $this->hasMany('\App\Invitation', 'invited_account_id');
    }

    public function getPendingInvitationCount()
    {
        $invitations = $this->invitations()->where('status','=',Invitation::STATUS_PENDING)->has('account');
        $connectionsViaNewsletterCount = $this->relatedAccountConnectionsViaNewsletter()->count();
        return $connectionsViaNewsletterCount+($invitations->count());
    }

    public function removeAgentAccount()
    {
        return $this->relatedAccounts()->wherePivot('type', AccountConnection::AGENT);
    }

    public function removeSellerAccount()
    {
        return $this->relatedFromAccount()->wherePivot('type', AccountConnection::AGENT);
    }

    public function sendMailToUsers($template, $data = [], $fromSystemOrUser = false)
    {
        $data['toAccount'] = $this;
        if (count($this->users()->count()) == 0) {
            Log::warning('Account does not have users with mail address assigned', [$this, $template]);
        }

        foreach ($this->users as $user) {
            $user->sendMail($template, $data, $fromSystemOrUser);
        }
    }

    public function getCount()
    {
        $count = $this->getPendingInvitationCount() + $this->getPendingCouponCount();

        return $count;
    }

    public function getPendingCouponCount()
    {
        return count($this->coupons()->where('status', '=', Coupon::STATUS_PENDING)->get());
    }

    public function coupons()
    {
        return $this->hasMany('\App\Coupon', 'consumer_user_id');
    }

    /**
     * Calculate conversion rate
     *
     * @return float|int
     */
    public function getConversionRate()
    {
        $sharedCoupons = 0;
        $sharedCouponsDone = 0;
        $conversionRate = 0;
        $users = $this->users()->get();

        foreach ($users as $user) {
            $couponsDoneCount = Coupon::where('agent_user_id', '=', $user->id)
                ->whereIn('status', Coupon::STATUS_GROUP_FINISHED)
                ->count();
            $sharedCoupons += $this->getSharedCouponCount();
            $sharedCouponsDone += $couponsDoneCount;
        }
        if ($this->getSharedCouponCount() != 0 && $sharedCouponsDone != 0) {
            $conversionRate = 100 / $this->getSharedCouponCount() * $sharedCouponsDone;
        }

        return $conversionRate;
    }

    /**
     * FIXME: use query->count() to not load everything
     * @return int
     */
    public function getSharedCouponCount()
    {
        return count($this->getSharedCoupons());
    }

    /**
     * FIXME: return query
     *
     * @return Illuminate\Database\Eloquent\Builder|Illuminate\Database\Query\Builder
     */
    public function getSharedCoupons()
    {
        $users = $this->users()->get();
        $coupons = [];

        foreach ($users as $user) {
            $userCoupons = $user->getSharedCoupons();

            $coupons = array_merge($coupons, $userCoupons->toArray());
        }

        return $coupons;
    }

    /**
     * @return array of Coupons with status redeemed or higher
     */
    public function getReedeemedCoupons(){
        $redeemedCoupons = [];
        foreach ($this->getSharedCoupons() as $coupon){
            if(in_array($coupon['status'], Coupon::STATUS_GROUP_FINISHED)){
                $redeemedCoupons[] = $coupon;
            }
        }
        return $redeemedCoupons;
    }

    /**
     * returns number of redeemed coupons
     * @return int
     */
    public function countRedeemedCoupons(){
        return count($this->getReedeemedCoupons());
    }

    /**
     * FIXME: optimize
     *
     * @return int|Illuminate\Database\Eloquent\Builder|Illuminate\Database\Query\Builder
     */
    public function getSharedCouponsNettoSum()
    {
        $coupons = $this->getSharedCoupons();
        $nettoSum = 0;

        foreach ($coupons as $coupon) {
            $nettoSum += $coupon['netto'];
        }

        return $nettoSum;
    }

    public function couponsByAgent()
    {
        return Coupon::whereIn('agent_user_id', $this->getUserIds());
    }

    public function sellerCoupons()
    {
        $templateIds = $this->couponTemplates()->pluck('coupon_templates.id');
        return Coupon::whereIn('coupon_template_id', $templateIds);
    }

    public function sellerCalculateCouponsNetto()
    {
        return $this->sellerCoupons()->sum('netto');
    }

    public function save(array $options = [])
    {
        if ($this->isDirty('zip')) {
            $locality = (new GeodbMaster())->searchByPLZ($this->zip)->first();
            if ($locality) {
                $this->lat = $locality->GeodbCoordinate()->lat;
                $this->lon = $locality->GeodbCoordinate()->lon;
            } else {
                $this->lat = null;
                $this->lon = null;
                Log::warning('ZIP Code ' . $this->zip . ' not found in geo database ', ['account' => $this->id]);
            }
        }
        return parent::save($options);
    }

    /**
     * Get Profile URL
     *
     * @param $role Which role is currently represented?
     * @return string
     * @throws Exception
     */
    public function getProfileUrl($role)
    {
        switch ($role) {
            case User::SELLER:
                return route('show-seller-account', $this->id);
            case User::AGENT:
                return route('show-agent-account', $this->id);
            case User::CONSUMER:
                return route('show-consumer-account', $this->id);
            default:
                throw new Exception('Unsupported profile role link');
        }
    }

    public function getProfileLink($role)
    {
        $displayName = $this->getDisplayName();
        return new HtmlString('<a href="' . $this->getProfileUrl($role) . '">' . $displayName . '</a>');
    }

    /**
     * Render link with avatar and name
     * @param string $role
     */
    public function renderFullLink($role = null)
    {
        $view = view('partials.account', ['account' => $this, 'role' => $role, 'long' => false]);
        return new HtmlString($view->render());
    }


    public function getDisplayAddress($html = false)
    {
        if ($html) {
            return new HtmlString('<span class="address"><span class="address-line1">' . $this->street . ' ' . $this->housenumber .
                '</span> <span>&middot;</span> <span class="address-line2">' . $this->zip . ' ' . $this->city . '</span></span>');
        } else {
            return $this->street . ' ' . $this->housenumer . ' · ' . $this->zip . ' ' . $this->city;
        }
    }

    public function getDisplayCity($html = false)
    {
        if ($html) {
            return new HtmlString('<span class="address"><span class="address-line2">' . $this->zip . ' ' . $this->city . '</span></span>');
        } else {
            return $this->zip . ' ' . $this->city;
        }
    }

    /**
     * SELLER: Get collection of current coupons that can be shared
     * @return array|Collection
     */
    public function getSellerShareableCoupons()
    {
        $result = new Collection();
        foreach ($this->couponTemplates as $template) {
            if ($template->isShareable()) {
                $result[] = $template;
            }
        }
        return $result;
    }

    /**
     * Get maxiumim cashback for all currently active coupons of a SELLER
     * @return int|mixed
     */
    public function getSellerMaxCashback()
    {
        $coupon = $this->getSellerShareableCoupons()->sortByDesc('discount_consumer')->first();
        if ($coupon) {
            return $coupon->discount_consumer;
        };

    }

    public function getSellerMaxRecoCash()
    {
        $coupon = $this->getSellerShareableCoupons()->sortByDesc('provision_agent')->first();
        if ($coupon) {
            return $coupon->provision_agent;
        }
    }

    /**
     * Creates a transaction based on the last day the payment was viable
     *
     * @return Invoice
     */
    public function createMembershipInvoice()
    {
        $user = $this->users()->first();

        // do not bill if billing period is not yet over
        if ($user->account->billed_until && $user->account->billed_until->gte(Carbon::today())) {
            return;
        }

        $billFrom = $user->account->billed_until ?? $this->account->received_seller_role_at ?? Carbon::today();

        $transactionDefaultData = [
            'user_id' => $user->id,
            'coupon_id' => null,
            'amount' => 0,
            'type' => null,
            'status' => Transaction::STATUS_ALLOCATED,
        ];

        $billUntil = clone $billFrom;

        if ($this->payment == 'month') {
            $billUntil->addMonth();

            $transaction = Transaction::create(array_merge($transactionDefaultData, [
                'amount' => config('recoma.monthlySubscriptionFee') * -1,
                'type' => Transaction::TYPE_SELLER_FEE_MONTHLY_PAYMENT,
                'billed_from' => $billFrom,
                'billed_until' => $billUntil,
            ]));

            $this->billed_until = $transaction->billed_until;
        } else {
            $billUntil->addYear();

            $transaction = Transaction::create(array_merge($transactionDefaultData, [
                'amount' => config('recoma.yearlySubscriptionFee') * -1,
                'type' => Transaction::TYPE_SELLER_FEE_YEARLY_PAYMENT,
                'billed_from' => $billFrom,
                'billed_until' => $billUntil,
            ]));

            $this->billed_until = $transaction->billed_until;
        }

        $this->save();

        $invoice = $this->createInvoice($transaction);

        return $invoice;
    }

    /**
     * Creates a transaction based on the last day the payment was viable and adds a month
     *
     * @return Invoice
     */
    public function createFreeMonthMembershipInvoice()
    {
        $user = $this->users()->first();

        $previousTransactions = Transaction::where('billed_until', '=', $user->account->billed_until)
            ->where('user_id', '=', $user->id)
            ->whereNull('coupon_id')
            ->get();

        $billedFrom = Carbon::today();

        if ($previousTransactions->count() > 0) {
            if ($previousTransactions->last()->billed_until != null) {
                $billedFrom = $previousTransactions->last()->billed_until;
            }
        }

        $transactionDefaultData = [
            'user_id' => $user->id,
            'coupon_id' => null,
            'amount' => 0,
            'type' => null,
            'status' => Transaction::STATUS_PAID,
        ];

        $billedUntil = clone $billedFrom;

        $billedUntil->addMonth();

        $transaction = Transaction::create(array_merge($transactionDefaultData, [
            'amount' => 0,
            'type' => Transaction::TYPE_SELLER_FREE_MONTHLY_PAYMENT,
            'billed_from' => $billedFrom,
            'billed_until' => $billedUntil,
        ]));

        $this->billed_until = $transaction->billed_until;

        $this->save();

        $invoice = $this->createInvoice($transaction);

        return $invoice;
    }

    /**
     * Creates the invoice for monthly/yearly payment
     *
     * @param $transaction
     * @return Invoice
     */
    public function createInvoice($transaction)
    {

        $invoice = new Invoice();
        $invoice->coupon_id = null;

        $invoice->transferInvoiceTransactionsToBillomat($transaction);
        $invoice->user_id = $transaction->user_id;

        $invoice->completeInvoice();

        $invoice->storePDF();

        $invoice->save();

        return $invoice;
    }

    /**
     * Sets date when role seller was given to a user from the account
     *
     * @param bool $isSeller
     */
    public function syncReceivedSellerRoleAt($isSeller = false)
    {
        $users = $this->users()->get();

        foreach ($users as $user) {
            if ($user->hasRole('seller')) {
                $isSeller = true;
            }
        }

        if ($this->received_seller_role_at == null && $isSeller == true) {
            $this->received_seller_role_at = Carbon::today();
            $this->save();
        }


        // alle user prüfen ob seller role
        // falls geändert,
        // dann setzen auf now, ansonsten null
    }

    public function createSepaMandat()
    {
        $user = $this->users()->first();
        $account = $this;
        $recoma = [
            'accountOwner' => config('banking.accountOwner'),
            'creditorId' => config('banking.creditorId'),
        ];

        $targetDir = storage_path('app/public/sepa/');
        if (!file_exists($targetDir)) {
            mkdir( $targetDir, 0770, true );
        }
        PDF::loadView('sepa.default', compact(['user', 'account', 'recoma']))
            ->save($targetDir . $account->id . '.pdf');

        $user->sendmail('sepa-mail', compact('user', 'pdf', 'account', 'recoma'), true, true);
    }

    /**
     * Determine if the account needs an onboarding
     * @return bool
     */
    public function needsOnBoarding()
    {
        if($this->users()->first()->hasRole('seller') && $this->received_seller_role_at == null && ($this->onboarding_process_level < OnboardingSellerController::MAX_LEVEL)){
            return true;
        }elseif($this->users()->first()->hasRole('agent') && $this->onboarding_process_level < OnboardingAgentController::MAX_LEVEL){
            return true;
        }
        return false;
    }

    /**
     * Logotype accessor.
     *
     * @return string
     */
    public function getLogoAttribute()
    {
        if ($this->upload_portrait) {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/' . $this->upload_portrait;
        } else {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/logo_company.png';
        }
    }

    /**
     * Logotype (200 x 200) accessor.
     *
     * @return string
     */
    public function getLogo200Attribute()
    {
        if ( file_exists(storage_path('app/portraits/') . $this->id . '_logo_200.png') ) {
            return env('APP_URL', 'https://reco.ma') . '/account/portraits/' . $this->id . '_logo_200.png';
        } else {
            return $this->logo;
        }
    }


    /**
     * Method to check whether user has filled minimal requirements for seller account.
     *
     * @return bool
     */
    public function hasSellerProfileFilled()
    {
        return
            $this->upload_portrait &&
            $this->company &&
            $this->hasAddress();
    }

    /**
     * Method to check whether user has address.
     *
     * @return bool
     */
    public function hasAddress()
    {
        return
            $this->street &&
            $this->housenumber &&
            $this->zip &&
            $this->city;
    }
}
